
#define LOG_SOURCE "jt"
#include <log.h>

#include <mempool.h>
#include <defs.h>
#include <workload.h>
#include <wltype.h>
#include <modules.h>
#include <modapi.h>
#include <randgen.h>

#include <jt.h>

#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/mman.h>

DECLARE_MODAPI_VERSION(MOD_API_VERSION);
DECLARE_MOD_NAME("jt");
DECLARE_MOD_TYPE(MOD_TSLOAD);

MODEXPORT wlp_descr_t jt_params[] = {
	{ WLP_INTEGER, WLPF_NO_FLAGS,
		WLP_NO_RANGE(),
		WLP_NO_DEFAULT(),
		"num_request_types",
		"Number of request types",
		offsetof(struct jt_workload, num_request_types) },
	{ WLP_INTEGER, WLPF_REQUEST,
        WLP_NO_RANGE(),
        WLP_NO_DEFAULT(),
        "request_type",
        "Request type",
        offsetof(struct jt_request, request_type) },
    { WLP_BOOL, WLPF_REQUEST,
        WLP_NO_RANGE(),
        WLP_NO_DEFAULT(),
        "is_incorrect",
        "Incorrect request",
        offsetof(struct jt_request, is_incorrect) },        
	{ WLP_NULL }
};

module_t* self = NULL;

static int process_request(void) {
    return -1;
}

static char req_letters[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_-";

static char* gen_request_name(uint64_t seed) {
    /* Use seed as follows: each letter is one of 64 values (6 bits)
       so use lower 4 bits for length, other bits as a letters */
    
    int length = (seed & 0xF) % 10;
    int i;
    
    char* name = (char*) mp_malloc(length + 1);
    
    seed = seed >> 4;
    
    for(i = 0; i < length; ++i) {
        name[i] = req_letters[seed & 0x3F];
        seed = seed >> 6;
    }
    
    name[i] = '\0';
    
    return name;
}

MODEXPORT int jt_wl_config(workload_t* wl) {
    struct jt_workload_data* data = mp_malloc(sizeof(struct jt_workload_data));
    struct jt_workload* jt_wl = (struct jt_workload*) wl->wl_params;
    
    int i;
    uint32_t idx;
    
    // int fd = open("/dev/zero", O_READ);
    
    randgen_t* rg;
    
    /* Allocate jump table */    
    data->jump_table = mmap(NULL, JT_SIZE, PROT_READ | PROT_WRITE, 
                            MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);    
    // close(fd);
    
    if(data->jump_table == (void*) (-1)) {
        wl_notify(wl, WLS_CFG_FAIL, -1, "Failed to allocate jump table");
        mp_free(data);
        return 1;
    }   
    
    wl_notify(wl, WLS_CONFIGURING, 25, "Allocated jump table");
           
    /* Generate request names */
    rg = rg_create(&rg_lcg_class, tm_get_clock());
    
    data->request_names = mp_malloc(sizeof(char*) * jt_wl->num_request_types);
    for(i = 0; i < jt_wl->num_request_types; ++i) {
        data->request_names[i] = gen_request_name(rg_generate_int(rg));
    }
    
    wl_notify(wl, WLS_CONFIGURING, 50, "Generated request names");
    
    rg_destroy(rg);
        
    for(i = 0; i < jt_wl->num_request_types; ++i) {
        idx = crc32s(0, data->request_names[i]);
        data->jump_table[idx] = process_request;
    }
    
    wl_notify(wl, WLS_CONFIGURING, 100, "Filled up jump table");
    
    wl->wl_private = data;
    
	return 0;
}

MODEXPORT int jt_wl_unconfig(workload_t* wl) {
    struct jt_workload_data* data = (struct jt_workload_data*) wl->wl_private;
    struct jt_workload* jt_wl = (struct jt_workload*) wl->wl_params;
    
    int i;    
    
    if(data) {
        for(i = 0; i < jt_wl->num_request_types; ++i) {
            mp_free(data->request_names[i]);
        }    
        
        mp_free(data->jump_table);
        munmap(data->jump_table, JT_SIZE);
        
        mp_free(data);
    }
    
	return 0;
}

MODEXPORT int jt_run_request(request_t* rq) {
    struct jt_workload_data* data = (struct jt_workload_data*) rq->rq_workload->wl_private;
    struct jt_workload* jt_wl = (struct jt_workload*) rq->rq_workload->wl_params;
	struct jt_request* jt_rq = (struct jt_request*) rq->rq_params;

    jt_func func = NULL; 
    char* name;
    uint32_t idx;
    
    if(jt_rq->is_incorrect) {
        name = gen_request_name(jt_rq->request_type);                
    }
    else {
        jt_rq->request_type = ((uint64_t) jt_rq->request_type) % jt_wl->num_request_types;
        name = data->request_names[jt_rq->request_type];       
    }
    
    idx = crc32s(0, name);
    func = data->jump_table[idx];
    if(func != NULL) {
        func();
    }
    
    if(jt_rq->is_incorrect) {
        mp_free(name);
    }
    
	return 0;
}

wl_type_t jt_wlt = {
	"jt",						/* wlt_name */

	WLC_CPU_MISC,						/* wlt_class */

	jt_params,					/* wlt_params */
	sizeof(struct jt_workload),	/* wlt_params_size*/
	sizeof(struct jt_request), 	/* wlt_rqparams_size*/

	jt_wl_config,				/* wlt_wl_config */
	jt_wl_unconfig,				/* wlt_wl_unconfig */

	NULL,								/* wlt_wl_step */

	jt_run_request				/* wlt_run_request */
};

MODEXPORT int mod_config(module_t* mod) {
	self = mod;

	wl_type_register(mod, &jt_wlt);

	return MOD_OK;
}

MODEXPORT int mod_unconfig(module_t* mod) {
	wl_type_unregister(mod, &jt_wlt);

	return MOD_OK;
}
